/*
 * Written by Dawid Kurzyniec, based on j.u.c.Condition by Doug Lea,
 * and released to the public domain, as explained at
 * http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.remote.locks;

import edu.emory.mathcs.backport.java.util.concurrent.*;
import java.util.Date;
import java.rmi.*;

/**
 * Interface mirroring the java.util.concurrent.locks.Condition,
 * but allowing for RemoteExceptions representing network communication errors.
 *
 * @author Dawid Kurzyniec
 */
public interface RemoteCondition extends Remote {

    /**
     * Causes the current thread to wait until it is signalled or
     * {@link Thread#interrupt interrupted}.
     *
     * @see edu.emory.mathcs.backport.java.util.concurrent.locks.Condition#awaitUninterruptibly
     *
     * @throws InterruptedException if the current thread is interrupted (and
     *         interruption of thread suspension is supported).
     * @throws RemoteException if a communication error occurs
     */
    void awaitUninterruptibly() throws RemoteException;

    /**
     * Causes the current thread to wait until it is signalled or interrupted,
     * or the specified waiting time elapses.
     *
     * @see edu.emory.mathcs.backport.java.util.concurrent.locks.Condition#awaitUninterruptibly
     *
     * @throws InterruptedException if the current thread is interrupted (and
     *         interruption of thread suspension is supported).
     * @throws RemoteException if a communication error occurs
     */
    boolean await(long time, TimeUnit unit) throws InterruptedException, RemoteException;

    /**
     * Causes the current thread to wait until it is signalled or interrupted,
     * or the specified deadline elapses.
     *
     * @see edu.emory.mathcs.backport.java.util.concurrent.locks.Condition#awaitUntil
     *
     * @throws InterruptedException if the current thread is interrupted (and
     *         interruption of thread suspension is supported).
     * @throws RemoteException if a communication error occurs
     */
    boolean awaitUntil(Date deadline) throws InterruptedException, RemoteException;

    /**
     * Wakes up one waiting thread.
     *
     * @see edu.emory.mathcs.backport.java.util.concurrent.locks.Condition#signal
     *
     * @throws RemoteException if a communication error occurs
     */
    void signal() throws RemoteException;

    /**
     * Wakes up all waiting threads.
     *
     * @see edu.emory.mathcs.backport.java.util.concurrent.locks.Condition#signalAll
     *
     * @throws RemoteException if a communication error occurs
     */
    void signalAll() throws RemoteException;
}
